/***********************************************************************
 *
 *	GSTI - A scripting engine for GNU Smalltalk
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 2005
 * Mike Anderson, gsti@gingerbread.plus.com
 *
 * This file is part of GSTI, a scripting engine for GNU Smalltalk
 *
 * GSTI is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later 
 * version.
 * 
 * GSTI is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GSTI; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 *
 ***********************************************************************/


#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include "getopt.h"
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>

char * host_name = "127.0.0.1";
int port = 5432;

static const struct option long_options[] = {
  {"eval", 1, 0, 'e'},
  {"server", 1, 0, 's'},
  {"port", 1, 0, 'p'},
  {"help", 0, 0, 'h'},
  {"version", 0, 0, 'v'},
  {NULL, 0, 0, 0}
};

static const char help_text[] =
  "mgst usage:"
  "\n"
  "\n    mgst [ -sp ] { script.st | -e <script> } arguments ..."
  "\n    mgst -vh"
  "\n"
  "\n   -e \t\t Evaluate argument as Smalltalk code"
  "\n   -s \t\t Specify a different server (default is localhost) *"
  "\n   -p \t\t Specify a different port (default is 5432) *"
  "\n   -v \t\t Print the version of the client and the server *"
  "\n   -h \t\t Print this message and exit\n"
  "\n   Options marked v, h, s and p are not yet implemented (sorry).\n";

typedef struct {
    int size;
    char **list;
} Args;

typedef struct {
    char script_type; /* 'e' or 'f' */
    char *value; /* script or filename */
    Args *args;
} Script;

int number_of_digits(int in)
{
    int i, n;
    n = 0;
    for (i = in; i > 0; i = i / 10, n++) ;
    return i;
}

Script *add_script_or_file(char script_or_file, char *a_value)
{
    char *add;
    int n, n1;

    Script *result;
    result = (Script *)malloc(sizeof(Script));
    result -> script_type = script_or_file;
    
    result -> value = (char *)malloc(strlen(a_value));
    strcpy(result -> value, a_value);
    
    /*printf("Adding script...\n");
    n = strlen(script);
    n1 = n + number_of_digits(n) + 10;
    add = (char *)malloc(n1);
    if (add == NULL) {
	perror("Not enough memory to store a copy of the script / filename?!");
	exit(1); 
    }
    *add = script_or_file;
    snprintf(add + 1, n1, " %d %s\000", n, script);
    printf(">%s<\n", add);
    todo[n_todo - 1] = add;
    printf("...\n", add);
    printf(">%c<\n", todo[n_todo - 1][0]);
    printf(">%d<\n", strlen(todo[n_todo - 1] + 1)); */
    
    return result;
}

int
parse_args (int argc,
	    char *argv[],
	    Script **out_script)
{
  int ch;
  Script *script = NULL;
  /* args = (Args *)malloc(sizeof Args); */
    
  /* get rid of getopt's own error reporting for invalid options */
  opterr = 1;
    
  while ((ch =
	  getopt_long (argc, (char **) argv, "+e:s:p:vh", long_options, NULL)) != -1)
    {

#if DEBUG_GETOPT
      printf ("%c \"%s\"  %d  %d  %d\n", ch, optarg ? optarg : "",
	      optind, prev_optind, minus_a_optind);
#endif

      switch (ch)
	{
	case 'e':
	  /* printf("-e argument found\n"); */
	  if (script) {
	    perror("Only one eval script, please.");
	    exit(1);
	  }
	  script = add_script_or_file('e', optarg);
	  break;

	default:
	  /* Fall through and show help message */
	case 'h':
	  printf (help_text);
	  return ch == 'h' ? -1 : 1;
	}
    }

    if (!script) {
	if (optind < argc) {
	    script = add_script_or_file('f', argv[optind]);
	    optind++;
	} else {
	    if (argc == 0) {
		printf (help_text);
		return(-1);	
	    } else {
		perror("No script supplied.");
		exit(1);
	    }
	}
    }
    
    *out_script = script;
    return 0;
}

int send_line(int socket, char line_type, char *line) {
    int result, n, m;
    char buf[8192];
    
    n = strlen(line);
    m = snprintf(
	buf, 
	8192, 
	"%c%d %s", 
	line_type, 
	n, 
	line
    );
    /* printf("Sending: >%s<\n", buf); */
    return send(socket, buf, m, 0);
}

int send_args(int socket, char *argv[], int from_index, int to_index) {
    int result, i;
    
    for (i = from_index; i <= to_index; i++) {
	result = send_line(socket, 'a', argv[i]);
	if (result == -1) { 
	    return -i;
	}
    }
    return 0;
}

int send_eval_script(int socket, Script *script) {
    return send_line(socket, 's', script -> value);
}

void try_restart_server() {
    int i;
    pid_t child_pid;
    char *st_image, *st_script;
    
    /* fprintf(stderr, "Restarting server.\n"); */
    i = system("pkill gsti-server");
    switch (WEXITSTATUS(i)) {
	/* Return values 0, 1 are OK - what does it mean if this fails? */
	case 0:
	case 1: 
	    break;
	default:
	    fprintf(stderr, "Error (%d) killing gsti-server.\n", i);
	    exit(1);
    }
    child_pid = fork();
    if (child_pid < 0) {
	perror("Failed to fork gsti-server");
	exit(1);
    }
    if (child_pid == 0) {
	/* Detach and spawn gsti-server */
	/* daemon(0, 1); */
	        
        /* Create a new SID for the child process */
        if (setsid() < 0) {
	    perror("Failed to set SID.");
	    exit(1);
        }

	if ((chdir("/")) < 0) {
	    perror("Failed to change to root dir");
	    exit(1);
	}
	
        umask(0);
	
	freopen("/tmp/gsti-server.log", "a", stdout);
	freopen("/tmp/gsti-server.log", "a", stderr);
	fclose(stdin);

	st_image = getenv("GSTI_IMAGE");
	st_script = getenv("GSTI_SCRIPT");
	execlp("gst", "gst", "-gI", st_image, st_script, NULL);
	perror("Failed to start gsti-server");
	/*fprintf(stderr, "(%s)", cmd);*/
	exit(1);
    } 
}

int main(int argc, char *argv[]) {
    char buf[8192];
    char message[256];
    int socket_descriptor;
    struct sockaddr_in pin;
    struct hostent *server_host_name;

    Script *script;

    int i, connect_result, n, send_result;

    /* printf("Starting...\n"); */
    
    parse_args(argc, argv, &script);

    /* printf("Parsed args...\n"); */
    
    /* Connect to server */
    if ((server_host_name = gethostbyname(host_name)) == 0) {
	perror("Error resolving local host");
	exit(1);
    }
    
    bzero(&pin, sizeof(pin));
    pin.sin_family = AF_INET;
    pin.sin_addr.s_addr = htonl(INADDR_ANY);
    pin.sin_addr.s_addr = ((struct in_addr * )(server_host_name -> h_addr)) -> s_addr;
    pin.sin_port = htons(port);
    	
    if ((socket_descriptor = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
	perror("Error opening socket (tried restarting server)");
	exit(1);
    }
    
    if (connect(socket_descriptor, (void *)&pin, sizeof(pin)) == -1) {
	try_restart_server();
	/* Wait for up to 3s, polling every 300ms */
	/* FIXME - make this configurable */
	for (i = 0; i < 10; i++) {
	    usleep(300000);  
	    if ((connect_result = connect(socket_descriptor, (void *)&pin, sizeof(pin))) != -1) {
		break;
	    }
	}
	if (connect_result == -1) {
	    perror("Error connecting to socket");
	    exit(1);
	}
    }

    /* Send scripts */	
    if (optind < argc) {
	if (send_args(socket_descriptor, argv, optind, argc - 1) < 0) {
	    perror("Error sending arguments");
	    exit(1);
	}
    }
    switch (script -> script_type) {
	case 'e' :
	    if (send_eval_script(socket_descriptor, script) == -1) {
		perror("Error sending eval script");
		exit(1);
	    }
	    break;
	case 'f' :
	    perror("Files are not yet implemented either");
	    exit(1);
	    /* send_result = send_file(socket_descriptor, script);
	    if (send_result == -1) {
		perror("Error sending script\n");
		exit(1);
	    }*/
	    /*send_result = send(socket_descriptor, todo[i] + 1, strlen(todo[i] - 1), 0);*/
	    break;
    }
	
    /* printf("waiting for response...\n"); */
    
    if (recv(socket_descriptor, buf, 8192, 0) == -1) {
	perror("Error in receiving response from server");
	exit(1);
    }
    
    printf("%s", buf);
    
    close(socket_descriptor);
}
