"***********************************************************************
 *
 *	GSTI - A scripting engine for GNU Smalltalk
 *
 *
 ***********************************************************************"

"***********************************************************************
 *
 * Copyright 2005
 * Mike Anderson, gsti@gingerbread.plus.com
 *
 * This file is part of GSTI, a scripting engine for GNU Smalltalk
 *
 * GSTI is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later 
 * version.
 * 
 * GSTI is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GSTI; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 *
 ***********************************************************************"


PackageLoader fileInPackages: #('TCP' "'MUtility' 'Compiler'")!

Smalltalk addSubspace: #GSTI!

Namespace current: GSTI!

Object subclass: #Server
    instanceVariableNames: 'port socket'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Net-GSTI'!
    
Object subclass: #Transaction
    instanceVariableNames: 'socket arguments'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Net-GSTI'!

Stream subclass: #SocketStream
    instanceVariableNames: 'socket pos'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Net-GSTI'!

SocketStream comment: 
'I provide the stream protocol methods that Sockets don''t support on their own.'
!

TextCollector subclass: #MultiplexingTranscript
    instanceVariableNames: 'default outputs'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Net-GSTI'!

MultiplexingTranscript comment: 
'Provides a separate Transcript for each process that registers with it.'
!

MultiplexingTranscript class methodsFor: 'instance creation'!

new
    ^(self basicNew) initialize; yourself
!
!

MultiplexingTranscript methodsFor: 'initialization'!

initialize
    super initialize.
    outputs := LookupTable new.
    Smalltalk at: Transcript put: self.
!
!

MultiplexingTranscript methodsFor: 'registering'!

register: aOutputStream
    semaphore critical:
	[ outputs at: Processor activeProcess put: aOutputStream ].
!

unregister
    semaphore critical:
	[ outputs removeKey: Processor activeProcess ].
!

primNextPutAll: aString
    | assoc |
    assoc := outputs at: Processor activeProcess ifAbsent: 
	[ FileStream stdout -> #nextPutAllFlush: ].
    assoc key perform: assoc value with: aString.
!
!

"Fix basicPrint to print to the Transcript, not stdout."

Object methodsFor: 'built ins'!

basicPrint
    Transcript 
	show: 'Object: an instance of '; 
	show: self class name; 
	show: '(';
	print: self asOop;
	show: ')';
	flush.
!
!

SmallInteger methodsFor: 'built ins'!

basicPrint
    Transcript show: 'Object: '; print: self; flush.
!
!

True methodsFor: 'built ins'!

basicPrint
    Transcript show: 'Object: true'.
!
!

False methodsFor: 'built ins'!

basicPrint
    Transcript show: 'Object: false'.
!
!

UndefinedObject methodsFor: 'built ins'!

basicPrint
    Transcript show: 'Object: nil'.
!
!

Character methodsFor: 'built ins'!

basicPrint
    Transcript show: 'Object: '; print: self.
!
!

Float methodsFor: 'built ins'!

basicPrint
    Transcript show: 'Object: '; print: self.
!
!

Symbol methodsFor: 'built ins'!

basicPrint
    Transcript show: 'Object: '; print: self.
!
!

CharacterArray methodsFor: 'built ins'!

basicPrint
    Transcript show: 'Object: '; print: self.
!
!

Class methodsFor: 'built ins'!

basicPrint
    Transcript show: 'Object: '; show: self name.
!
!

Object methodsFor: 'built ins'!

primError: message

    "This might start the debugger... Note that we use #basicPrint
     'cause #printOn: might invoke an error."
     
    "Overridden to use the [Multiplexing]Transcript instead of stdout."

    | debuggerClass context |
    "Transcript initialize. - Don't see that this is appropriate."
    Transcript flush.
    debuggerClass := thisContext debuggerClass.
    debuggerClass isNil ifFalse: [
	^debuggerClass
	    openOn: Processor activeProcess
	    message: self class printString, ' error: ', message
    ].

    "Default behavior - print backtrace"
    RegressionTesting ifFalse: [ self basicPrint ].
    Transcript
	nextPutAll: ' error: ';
	display: message;
	nl.

    RegressionTesting ifFalse: [
        context := thisContext.
        "[ context isInternalExceptionHandlingContext ]
            whileTrue: [ context := context parentContext ]."

        context backtraceOn: Transcript
    ].

    Transcript flush.
    ContextPart unwind
!
!

SocketStream class methodsFor: 'instance creation'!

on: aSocket
    ^(self new) socket: aSocket.
!
!

SocketStream methodsFor: 'accessing'!

socket: aSocket
    socket := aSocket.
    pos := 1.
!
!

SocketStream methodsFor: 'decorator'!

skip: aCount
    aCount = -1 ifTrue: [ ^self ].
    ^super skip: aCount
!

next
    | ch |
    pos := pos + 1.
    [ socket available ] whileFalse: [ Processor yield ].
    ch := socket next.
    "Transcript << $> << ch."
    ^ch
!

peek
    | ch |
    [ socket available ] whileFalse: [ Processor yield ].
    ch := socket peek.
    "Transcript << $? << ch."
    ^ch
!

peekFor: aCharacter
    ^socket peekFor: aCharacter
!

position
    ^pos
!

atEnd
    ^socket atEnd
!
!

Server class methodsFor: 'instance creation'!

new
    ^(self basicNew) initialize; yourself
!
!

Server methodsFor: 'initialization'!

initialize
    port := 5432.
!
!

Transaction class methodsFor: 'instance creation'!

new
    ^(self basicNew) initialize; yourself
!
!

Transaction methodsFor: 'initialization'!

initialize
    ^self
!
!

Transaction methodsFor: 'accessing'!

socket: aSocket
    socket := aSocket.
!
!

Server methodsFor: 'running'!

run
    socket := TCP.ServerSocket port: port.
    Transcript << 'GSTI server starting...'; nl.
    
    [ | trans |
	Transcript << 'Waiting for connection...'; nl.
	socket waitForConnection.
	Transcript << 'New connection - forking Transaction.'; nl.
	trans := (Transaction new) socket: socket accept.
	[ trans run ] fork.
	Processor yield. ] 
	repeat.
    socket close.
!
!

Transaction methodsFor: 'running'!
      
readLineFrom: aStream
    "Read a 'line' from the socket, beginning with its size."
    
    | n line |
    
    n := Integer readFrom: aStream.
    Transcript << 'Message in: ' << n.
    
    line := WriteStream on: String new.
    n timesRepeat: [ line nextPut: aStream next ].
    Transcript << ' ' << line contents; nl.
    ^line contents
!

run
    | out sm ch script sent sender |
    Transcript << 'Transaction starts...'; nl.

    [ socket available ] whileFalse: [ Processor yield ].
    
    sm := SocketStream on: socket.
    
    arguments := OrderedCollection new.
    [ 	ch := socket next.
	ch = $a ]
	whileTrue:
	[ arguments add: (self readLineFrom: sm) ].
    
    ch = $s ifFalse: [ self error: 'Expecting the script now, but found: ', ch asString ].

    script := self readLineFrom: sm.
    
    sent := false.
    sender := [ :msg | sent := true. socket nextPutAll: msg; flush ].
    
    Transcript register: sender -> #value:.
    out := Behavior 
	evaluate: script
	to: self
	ifError:
	    [ :fname :line :msg |
	    Transcript << 'Error at line ' << line printString << ': ' << msg << '(' << fname printString << ')'. ].
    Transcript unregister.

    sent ifFalse: [ sender value: out printString ].

    [ socket close. ] on: Error do: [ :sig | Transcript << 'Ignoring error on socket close.' ; nl. ].
    Transcript << 'Transaction complete.'; nl.
!
!

Transcript become: MultiplexingTranscript new.
Server new run.
!

"Object primError:"